package infinitebuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.LinkedBlockingQueue;

import base.Arrival;
import base.Deterministic_I;


/**
 * This class is created for testing of priority based scheduling mechanism 
 * with removal process with real arrivals and timeouts which does not use
 * analytical methods. Tt creates retransmission timers for original and 
 * retransmitted arrivals to verify the retransmission count consistency 
 * at a time-slot
 *  
 * @author demiry
 *
 */
public class PrioRemovalTest {
   public static int intervalLength = 50; // 50ms
   public static double arrivalRate = 200;
   public static double serviceRate = 1000;
   
   public static int simulationTime = 50000; // 50seconds
   /* simulation time on time-intervals */
   public static int totalTime = (int) (simulationTime/intervalLength); 
   
   public static double t1 = 500; // 500ms
   public static double t1_2 = t1*2;
   public static double t1_3 = t1_2*2;
   public static double t1_4 = t1_3*2;
   public static double t1_5 = t1_4*2;
   public static double t1_6 = t1_5*2;
   
   public static int t1n   = (int)t1/(int)intervalLength;
   public static int t1_2n = (int)t1_2/(int)intervalLength;
   public static int t1_3n = (int)t1_3/(int)intervalLength;
   public static int t1_4n = (int)t1_4/(int)intervalLength;
   public static int t1_5n = (int)t1_5/(int)intervalLength;
   public static int t1_6n = (int)t1_6/(int)intervalLength;
   
   public static int nt1 = (int)t1/(int)intervalLength;
   public static int nt1_2 = nt1 + (int)t1_2/(int)intervalLength;
   public static int nt1_3 = nt1_2 + (int)t1_3/(int)intervalLength;
   public static int nt1_4 = nt1_3 + (int)t1_4/(int)intervalLength;
   public static int nt1_5 = nt1_4 + (int)t1_5/(int)intervalLength;
   public static int nt1_6 = nt1_5 + (int)t1_6/(int)intervalLength;
   
   public int queueSizeInitial = 5500;
   public int arrivalRateInitial = queueSizeInitial;
   
   public ArrayList<Integer> queueSizes = null;
   public ArrayList<Integer> arrivals   = null;
   public ArrayList<Integer> services   = null;
   public ArrayList<Integer> secQueueSizes = null; /* retransmission queue */
   public ArrayList<Integer>[] rtsizes = null; 
   
   public LinkedBlockingQueue<Arrival> queue = new LinkedBlockingQueue<Arrival>();
   public LinkedBlockingQueue<Arrival> retransQueue = new LinkedBlockingQueue<Arrival>();
   public int arrivalIndex = 0;
   public ArrayList<ArrayList<Arrival>> retransTimers = new ArrayList<ArrayList<Arrival>>(); 
   
   public boolean debug = true;
   public boolean debugX = false;
   
   public int[] currentRetransCounts = new int[6];
   public int[] currRemovedCounts = new int[6];
      
   private class PositionData {
      public int timeslot = 0;
      public int position = 0;
      
      public PositionData() {}
      
      public PositionData(int slot, int pos) {
         this.timeslot = slot;
         this.position = pos;
      }
      
      public String toString() {
         return "ts=" + this.timeslot + ", pos=" + this.position;
      }
   }
   
   public PositionData currentPosition = new PositionData();
      
   private void dumpPrimaryQueue() {
      System.out.println("-------Primary queue dump with size=" + this.queue.size());
      Iterator<Arrival> it = queue.iterator();
      while (it.hasNext()) {
         Arrival arr = it.next();
         System.out.print("[" + arr + "]");
      }
      System.out.println(".");
   }
   
   private void dumpSecondaryQueue() {
      System.out.println("-------Secondary queue dump with size=" + this.retransQueue.size());
      Iterator<Arrival> it = retransQueue.iterator();
      while (it.hasNext()) {
         Arrival arr = it.next();
         System.out.print("[" + arr + "]");
      }
      System.out.println(".");
   }

   private int getNextRetransmitTime(Arrival arr, int n) {
      switch (arr.type) {
         case Arrival.NEW_ARRIVAL:
            arr.retTime = n+t1n;
            return Arrival.FIRST_RETRANS;
            
         case Arrival.FIRST_RETRANS:
            arr.retTime = n+t1_2n;
            return Arrival.SECOND_RETRANS;
            
         case Arrival.SECOND_RETRANS:
            arr.retTime = n+t1_3n;
            return Arrival.THIRD_RETRANS;
            
         case Arrival.THIRD_RETRANS:
            arr.retTime = n+t1_4n;
            return Arrival.FOURTH_RETRANS;
            
         case Arrival.FOURTH_RETRANS:
            arr.retTime = n+t1_5n;
            return Arrival.FIFTH_RETRANS;
            
         case Arrival.FIFTH_RETRANS:
            arr.retTime = n+t1_6n;
            return Arrival.SIXTH_RETRANS;
            
         case Arrival.SIXTH_RETRANS:
            return arr.type=-1;
      }
      return -1;
   }

   public void init() {
      arrivals = Deterministic_I.obtainArrivals((arrivalRate * intervalLength)/1000.0, totalTime);
      services = Deterministic_I.obtainServiceRates((serviceRate * intervalLength)/1000.0, totalTime);
      queueSizes = new ArrayList<Integer>(totalTime+1);
      queueSizes.add(queueSizeInitial);
      this.secQueueSizes = new ArrayList<Integer>(totalTime+1);
      this.secQueueSizes.add(0);
      this.rtsizes = new ArrayList[7];
      for (int i=0; i<7; i++) {
         this.rtsizes[i] = new ArrayList<Integer>(totalTime+1);
         this.rtsizes[i].add(0);
      }
      for (int i=0; i<6; i++) {
         retransTimers.add(new ArrayList<Arrival>());
      }
      for (int i=0; i<queueSizeInitial; i++) {
         Arrival arr = new Arrival(i, 0, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, 0);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
      }
      arrivals.set(0, 0);
   }
      
   private int getArrivalCount(int n) {
      int dcount = arrivals.get(n);
      int count = (int)dcount;
      if (this.debug) {
         System.out.print("ARRIVALS[" + n + "]: ");
      }
      for (int i=0; i<count; i++) {
         Arrival arr = new Arrival(arrivalIndex, n, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, n);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
         if (this.debug) {
            System.out.print(arr.id + ":" + arr.type + ":" + arr.retTime + ", ");
         }
      }
      if (this.debug) {
         System.out.println(" ...");
      }
      return dcount;
   }
   
   private void stopRetransmitTimer(Arrival arr) {
      for (int i=0; i<retransTimers.size(); i++) {
         ArrayList<Arrival> rta = retransTimers.get(i);
         boolean res = rta.remove(arr);
         if (this.debug) {
            if (res) {
               System.out.print("+");
            }
            else {
               System.out.print("-");
            }
         }
      }
   }
   
   private int removeFromRetransmitQueue(Arrival arr) {
      int count = 0;

      if (debug) {
         Iterator<Arrival> it = retransQueue.iterator();
         while (it.hasNext()) {
            Arrival tarr = it.next();
            if (tarr.id == arr.id) {
               if (this.debug) {
                  System.out.print("(" + tarr + ")");
               }
               count++;
               it.remove();
               currRemovedCounts[tarr.type-1]++;
            }
         }
      }
      else {
         while (retransQueue.remove(arr)) {
            if (this.debug) {
               System.out.print("(" + arr + ")");
            }
            count++;
         }
      }
      if (this.debug) {
         System.out.print("(" + count + ")");
      }
      return count;
   }
   
   private int getServiceCount(int n) {
      int dsrvcount = services.get(n);
      int srvcount = (int)dsrvcount;
      int removedRetrans = 0;
      if (this.debug) {
         System.out.print("SERVICE: ");
      }
      for (int i=0; i<srvcount; i++) {
         Arrival arr = queue.poll();
         if (arr != null) {
            if (this.debug) {
               System.out.print(arr);
            }
            stopRetransmitTimer(arr);
            removedRetrans += removeFromRetransmitQueue(arr);
            if (this.debug) {
               System.out.print(", ");
            }
         }
         else {
            if (this.debug) {
               System.out.print("Server-Idle ");
            }
         }
      }
      if (this.debug) {
         System.out.println(" ... <" + removedRetrans + ">");
      }
      return dsrvcount;
   }
   
   private int calcInternalRetrans(int n, int t1Count) {
//      if (n==23) {
//         System.out.println("N=" + n);
//      }
      int val1 = 0;
      int val2 = 0;
      if ((n-t1Count) >= 0) {
         //for (int i=1; i<=t1Count; i++) {
         for (int i=0; i<=t1Count; i++) {
            int index = n-t1Count + i;
            if (this.debugX  && (t1Count < 15)) {
               System.out.println("services[" + index + "]= " + services.get(index));
            }
            val1 += services.get(index);
         }
         if ((n-t1Count) == 0) {
            val1 = queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = queueSizes.get(n-t1Count); // Assume arrivals at time 0
         }
         else {
            if (this.debugX  && (t1Count < 15)) {
               System.out.println("arrivals[" + (n-t1Count) + "]= " + arrivals.get(n-t1Count));
               System.out.println("queueSizes[" + (n-t1Count) + "]= " + queueSizes.get(n-t1Count));
            }
            val1 = arrivals.get(n-t1Count) +  queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = arrivals.get(n-t1Count);
         }
      }
      if (this.debugX  && (t1Count < 15)) {
         System.out.println("val1= " + val1 + " val2= " + val2);
      }
      return (val1 > val2) ? val2 : val1;
   }

   private void checkForSanity(int index, int value, int n) {
      ArrayList<Arrival> rta = retransTimers.get(index-1);
      int count = 0;
      if (this.debug) {
         System.out.print("RT-arrivals<" + index + ">: ");
      }
      while (!rta.isEmpty()) {
         Arrival arr = rta.get(0);
         if (arr == null) {
            break;
         }
         if (arr.retTime != n) {
            break;
         }
         arr = rta.remove(0);
         //Arrival newArr = new Arrival(arr.id, n, arr.type+1);
         Arrival newArr = new Arrival(arr, n);
         //newArr.type++;
         int rt = getNextRetransmitTime(newArr, n);
         /* retTime has been updated by getNextRetransmitTime */
         //newArr.type++;
         //newArr.n = n;
         ArrayList<Arrival> rta1 = retransTimers.get(rt-1);
         rta1.add(newArr);
         retransQueue.offer(newArr);
         if (this.debug) {
            System.out.print(newArr.id + ":" + newArr.type + ":" + newArr.retTime + ", ");
         }
         count++;
      }
      if (this.debug) {
         Object[] arrTable = queue.toArray();
         System.out.println(" size=" + arrTable.length);
         System.out.println(" ...");
      }
      if (count != value) {
         System.err.println("ERROR: Estimated retransmit count<" + value + "> and found count<" + count + "> are different!!!");
      }
   }
   
   private int getRetransCount(int n) {
      /* r(n) += rj(n), j=1,..,6 
       * rj(n) = min{ [arr(n-Tj)+q(n-Tj) - (+)srv(n-Tj+k), k=1,..,Tj; j=1,..,6]positive, arr(n-Tj) }
       */
      int r = 0;
      int r1 = calcInternalRetrans(n, nt1);
      if (n-nt1 >= 0) {
         this.rtsizes[0].add((n-nt1), r1);
      }
      else {
         this.rtsizes[0].add(r1);
      }
      checkForSanity(Arrival.FIRST_RETRANS, (int)r1, n);
      
      int r2 = calcInternalRetrans(n, nt1_2); 
      if (n-nt1_2 >= 0) {
         this.rtsizes[1].add((n-nt1_2), r2);
      }
      else {
         this.rtsizes[1].add(r2);
      }
      checkForSanity(Arrival.SECOND_RETRANS, (int)r2, n);
      
      int r3 = calcInternalRetrans(n, nt1_3);
      if (n-nt1_3 >= 0) {
         this.rtsizes[2].add((n-nt1_3), r3);
      }
      else {
         this.rtsizes[2].add(r3);
      }
      checkForSanity(Arrival.THIRD_RETRANS, (int)r3, n);

      int r4 = calcInternalRetrans(n, nt1_4);
      if (n-nt1_4 >= 0) {
         this.rtsizes[3].add((n-nt1_3), r4);
      }
      else {
         this.rtsizes[3].add(r4);
      }
      checkForSanity(Arrival.FOURTH_RETRANS, (int)r4, n);
      
      int r5 = calcInternalRetrans(n, nt1_5);
      if (n-nt1_5 >= 0) {
         this.rtsizes[4].add((n-nt1_3), r5);
      }
      else {
         this.rtsizes[4].add(r5);
      }
      checkForSanity(Arrival.FIFTH_RETRANS, (int)r5, n);
      
      int r6 = calcInternalRetrans(n, nt1_6);
      if (n-nt1_6 >= 0) {
         this.rtsizes[5].add((n-nt1_3), r6);
      }
      else {
         this.rtsizes[5].add(r6);
      }
      checkForSanity(Arrival.SIXTH_RETRANS, (int)r6, n);      
      
      r = r1 + r2 + r3 + r4 + r5 + r6;
      if (this.debug) {
         System.out.println("r[" + n + "]=" + r + " (r1=" + r1 + ", r2=" + r2 + ", r3=" + r3 + ", r4=" + r4 + ", r5=" + r5 + ", r6=" + r6 + ")");
      }
      this.currentRetransCounts[0] = r1;
      this.currentRetransCounts[1] = r2;
      this.currentRetransCounts[2] = r3;
      this.currentRetransCounts[3] = r4;
      this.currentRetransCounts[4] = r5;
      this.currentRetransCounts[5] = r6;

      return r;
   }
   
   private int getQueueSize(int n) {
      return queueSizes.get(n);
   }
   
   private ArrayList<PositionData> determinePositionData(int n) {
      ArrayList<PositionData> pd = new ArrayList<PositionData>();
      
      int servCount = this.services.get(n);
      this.currentPosition.position += servCount;
      if (this.currentPosition.timeslot == 0) {
         if (this.queueSizeInitial > 0) {
            if (this.currentPosition.position < this.queueSizeInitial) {
               pd.add(new PositionData(this.currentPosition.timeslot, servCount));
               return pd;
            }
            else {
               this.currentPosition.position = this.currentPosition.position - this.queueSizeInitial;
               int temp = servCount - this.currentPosition.position;
               pd.add(new PositionData(this.currentPosition.timeslot, temp));
               servCount = this.currentPosition.position;
               this.currentPosition.timeslot = 1;
            }
         }
      }
      for (int i=this.currentPosition.timeslot; i<=n; i++) {
         if (this.debug) {
            System.out.println("slot=" + this.currentPosition.timeslot + " position=" + this.currentPosition.position + 
                               " arrival[" + i + "]=" + this.arrivals.get(i) + " service[" + n + "]=" + this.services.get(n) +
                               ", service-remaining=" + servCount);
         }
         if (this.currentPosition.position >= this.arrivals.get(i)) {
            this.currentPosition.position = this.currentPosition.position - this.arrivals.get(i);
            int temp = servCount - this.currentPosition.position;
            pd.add(new PositionData(this.currentPosition.timeslot, temp));
            servCount = this.currentPosition.position;
            this.currentPosition.timeslot = i+1;
            if (i == n) {
               this.currentPosition.position = 0;
            }
         }
         else {
            pd.add(new PositionData(this.currentPosition.timeslot, servCount));
            break;
         }
      }
      return pd;
   }
   
   public void doFluidFlow() {
      int n = 0; // time interval
      //this.debug = false;
      while (n<totalTime) {
         if (this.debug) {
            System.out.println("n = " + n + " ------------------------------------------------------------------");
         }         
         if (this.debug) {
            dumpPrimaryQueue();
            dumpSecondaryQueue();
         }
         int queueSize = getQueueSize(n);
         if (this.debug) {
            System.out.println("current[" + n + "]=" + queueSize);
         }
         
         int arrCount  = getArrivalCount(n);
         if (this.debug) {
            System.out.println("lambda[" + n + "]=" + arrCount);
         }
         
         for (int i=0; i<6; i++) currRemovedCounts[i] = 0;
         
         int srvCount  = getServiceCount(n);
         if (this.debug) {
            System.out.println("mu[" + n + "]=" + srvCount);
         }
         
         int newQSize = queueSize + arrCount - srvCount;
         if (newQSize < 0) {
            newQSize = 0;
         }
         ArrayList<PositionData> pda = determinePositionData(n);
         if (this.debug) { 
            System.out.println("new timeslot [" + (n+1) + "]= " + this.currentPosition.timeslot + ", and position= " + this.currentPosition.position);
         }
         for (int i=0; i<pda.size(); i++) {
            PositionData pd = pda.get(i);
            if (this.debug) {
               System.out.println("pd[" + i + "]==> " + pd);
            }
         }
         
         if (this.debug) {
            System.out.println("primary-queue-size[" + (n+1) + "]=" + newQSize);
         }
         queueSizes.add(newQSize);
         
         int removeCount = 0;
         for (int i=0; i<pda.size(); i++) {
            PositionData pd = pda.get(i);
            for (int j=0; j<6; j++) {
               if ((pd.timeslot > nt1) && (rtsizes[j].get(pd.timeslot) > 0)) {
                  removeCount += pd.position;
               }
            }
         }
         if (this.debug) {
            System.out.println("Estimated removed count = " + removeCount);
         }
         
         int retCount  = getRetransCount(n);
         if (this.debug) {
            System.out.println("r[" + n + "]=" + retCount);
         }
         
         int calcRetransQueue = (this.secQueueSizes.get(n) - removeCount);
         if (calcRetransQueue < 0) {
            calcRetransQueue = 0;
         }
         calcRetransQueue += retCount;
         
         if ((calcRetransQueue > 0) && (newQSize == 0)) {
            //System.err.println("ERROR: Unexpected situation --> " + n);
         }
         
         if (this.debug) {
            System.out.println("Calculated rt-queue= " + calcRetransQueue);
         }
         this.secQueueSizes.add(calcRetransQueue);
         if (this.debug) {
            System.out.println("secondary queue-size[" + (n+1) + "]=" + calcRetransQueue);
         }
         
         n++;
      }
   }
   
   public static void main(String[] args) {
      PrioRemovalTest sim = new PrioRemovalTest();
      sim.init();
      sim.doFluidFlow();
   }
}
