package infinitebuffer;
import java.util.ArrayList;

import base.Deterministic_I;
import base.FluidFlowSim;

/**
 * Fluid-flow based simulation for priority based scheduling mechanism
 * with removal process. 
 * 
 * @author demiry
 *
 */
public class PrioRemoval extends FluidFlowSim {
   
   public ArrayList<Integer> secQueueSizes = null; /* retransmission queue */
   public ArrayList<Integer>[] rtsizes = null;
   
   public ArrayList<Integer> pArrivalTimeslot = null; // u(n)
   public ArrayList<Integer> pInnerPosition   = null; // v(n)
   public ArrayList<ArrayList<Integer>> pCounts = null; // Sn(u,v)
   
   public ArrayList<Integer> estimatedDelay   = null;
   
   public int lastproc = 0; /* estimated one */
      
   public void init() {
      this.arrivals = Deterministic_I.obtainArrivals((arrivalRate * intervalLength)/1000.0, totalTime);
      this.services = Deterministic_I.obtainServiceRates((serviceRate * intervalLength)/1000.0, totalTime);
      this.queueSizes = new ArrayList<Integer>(totalTime+1);
      this.queueSizes.add(queueSizeInitial);
      this.secQueueSizes = new ArrayList<Integer>(totalTime+1);
      this.secQueueSizes.add(0);
      this.pArrivalTimeslot = new ArrayList<Integer>(totalTime+1);
      this.pArrivalTimeslot.add(0);
      if (queueSizeInitial > 0) {
         this.pArrivalTimeslot.add(0); /* we need to start with initial-queue */
      }
      else {
         this.pArrivalTimeslot.add(1); /* trick way to start with arrival time slot 1 at processing time slot 1 */
      }
      this.pInnerPosition = new ArrayList<Integer>(totalTime+1);
      this.pInnerPosition.add(0);
      this.pInnerPosition.add(0); /* tricky way to start from 0 of starting arrival time slot */
      this.pCounts = new ArrayList<ArrayList<Integer>>();
      ArrayList<Integer> temp = new ArrayList<Integer>();
      temp.add(0);
      this.pCounts.add(temp);
      
      this.rtsizes = new ArrayList[7];
      for (int i=0; i<7; i++) {
         this.rtsizes[i] = new ArrayList<Integer>(totalTime+1);
         this.rtsizes[i].add(0);
      }
      this.arrivals.add(0, 0);
      this.services.add(0, 0);
      
      this.estimatedDelay = new ArrayList<Integer>(totalTime+1);
   }
         
   protected int checkForCeasedArrivals(int n) {
      int calcCeased = 0;
      calcCeased = calcInternalRetrans(n, nt1_6+t1n);
      
      return calcCeased;
   }
   
   public int getRetransCounts(int n) {
      /* r(n) += rj(n), j=1,..,6 
       * rj(n) = min{ [arr(n-Tj)+q(n-Tj) - (+)srv(n-Tj+k), k=1,..,Tj; j=1,..,6]positive, arr(n-Tj) }
       */
      int r = 0;
      int r1 = calcInternalRetrans(n, nt1);
      int r2 = calcInternalRetrans(n, nt1_2); 
      int r3 = calcInternalRetrans(n, nt1_3);
      int r4 = calcInternalRetrans(n, nt1_4);
      int r5 = calcInternalRetrans(n, nt1_5);
      int r6 = calcInternalRetrans(n, nt1_6);
      
      if (n-nt1 >= 0) {
         this.rtsizes[0].add((n-nt1), r1);
      }
      else {
         this.rtsizes[0].add(r1);
      }
      if (n-nt1_2 >= 0) {
         this.rtsizes[1].add((n-nt1_2), r2);
      }
      else {
         this.rtsizes[1].add(r2);
      }
      if (n-nt1_3 >= 0) {
         this.rtsizes[2].add((n-nt1_3), r3);
      }
      else {
         this.rtsizes[2].add(r3);
      }
      if (n-nt1_4 >= 0) {
         this.rtsizes[3].add((n-nt1_4), r4);
      }
      else {
         this.rtsizes[3].add(r4);
      }
      if (n-nt1_5 >= 0) {
         this.rtsizes[4].add((n-nt1_5), r5);
      }
      else {
         this.rtsizes[4].add(r5);
      }
      if (n-nt1_6 >= 0) {
         this.rtsizes[5].add((n-nt1_6), r6);
      }
      else {
         this.rtsizes[5].add(r6);
      }
      
      r = r1 + r2 + r3 + r4 + r5 + r6;
      if (this.debug) {
         System.out.println("r[" + n + "]=" + r + " (r1=" + r1 + ", r2=" + r2 + ", r3=" + r3 + ", r4=" + r4 + ", r5=" + r5 + ", r6=" + r6 + ")");
      }
      currentRetransCounts[0] = r1;
      currentRetransCounts[1] = r2;
      currentRetransCounts[2] = r3;
      currentRetransCounts[3] = r4;
      currentRetransCounts[4] = r5;
      currentRetransCounts[5] = r6;
      
      this.totalRetSeq[0] += r1;
      this.totalRetSeq[1] += r2;
      this.totalRetSeq[2] += r3;
      this.totalRetSeq[3] += r4;
      this.totalRetSeq[4] += r5;
      this.totalRetSeq[5] += r6;
      
      this.currRtrate += r;
      this.rtrateCounter++;
      //System.out.println("rtCounter = " + rtrateCounter + " Current rate[" + n + "] = " + this.currRtrate);
      if (this.rtrateCounter >= slotForSec) {
         this.rtrateCounterAll++;
         //System.out.println("Added retrans rate value = " + this.currRtrate + " with index " + this.rtrateCounterAll);
         retransRates.add(this.currRtrate);
         this.currRtrate = 0;
         this.rtrateCounter = 0;
      }
      return r;
   }
   
   private String printProcessedCounts(int timeslot) {
      if ((this.pCounts == null) || (this.pCounts.isEmpty())) return "";
      
      StringBuilder sb = new StringBuilder();
      ArrayList<Integer> pcList = this.pCounts.get(timeslot);
      if ((pcList == null) || (pcList.isEmpty())) return "";
      sb.append('{');
      int k = 0;
      if (timeslot > 0) {
         k = this.pArrivalTimeslot.get(timeslot);
      }
      for (int i=0; i<pcList.size(); i++) {
         sb.append("s(" + (k+i) + ")=" + pcList.get(i) + ", ");
      }
      sb.append('}');
      return sb.toString();
   }
   
   public ArrayList<Integer> determineProcessedArrivalSlots(int n) {
      ArrayList<Integer> localList = new ArrayList<Integer>();
      
      if (n == 0) {
         return localList;
      }
      int servCount = this.services.get(n);
      
      int currArrTs = this.pArrivalTimeslot.get(n);
      int currInner = this.pInnerPosition.get(n);
      
      if (debug) {
         System.out.println("~~~~~~~~~[" + n + "]~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
         System.out.println("Beginning: servPower=" + servCount + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
      }

      if (currArrTs == 0) {
         if (this.queueSizeInitial > 0) {
            /* Processing on initial-queue */
            currInner += servCount;
            
            if (currInner < this.queueSizeInitial) {
               /* still processing on initial-queue */
               localList.add(servCount);
               this.pArrivalTimeslot.add(currArrTs);
               this.pInnerPosition.add(currInner);
               this.pCounts.add(localList);
               
               if (debug) {
                  System.out.println("Still on the initial queue: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " S_" + n + "=" + printProcessedCounts(n));
               }
               return localList;
            }
            else if (currInner == this.queueSizeInitial) {
               /* Just finished the initial queue. We should continue with
                * the actual requests starting from the first arrival time slot 
                * at next processing time slot
                */
               localList.add(servCount);
               this.pArrivalTimeslot.add(1);
               this.pInnerPosition.add(0);
               this.pCounts.add(localList);
               
               return localList;
            }
            else {
               /* the processing of the initial-queue is just completed */
               /* there is a need to determine the processing for the actual queue */
               /* determine the last processed part from the initial-queue and
                * let the remaining to the following code-block for processing 
                */
               localList.add(servCount - (currInner - this.queueSizeInitial));
               servCount = currInner - this.queueSizeInitial; 
               currArrTs = 1;
               currInner = 0;
               if (debug) {
                  System.out.println("Initial queue is just completed: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
               }
            }
         }
      }
      
      /* Running on out of initial queue */
      for (int i=currArrTs; i<=n; i++) {
         int remain = this.arrivals.get(i) - currInner;
         if (debug) {
            System.out.println("Serving for timeslot: servPower=" + servCount + " remaining=" + remain + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
         }
         if (servCount < remain) {
            /* no change on processing timeslot but inner position */
            this.lastproc = i;
            currArrTs = i;
            currInner += servCount;
            localList.add(servCount);
            if (debug) {
               System.out.println("Still on timeslot = " + i + ": u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner );
            }
            break;
         }
         else if (servCount == remain) {
            /* promate to the next timeslot and initial position */
            this.lastproc = i;
            currArrTs = i+1;
            currInner = 0;
            localList.add(servCount);
            
            if (debug) {
               System.out.println("Service is just completed for timeslot = " + i + ": u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
            }
            break;
         }
         else {
            /* service power is bigger then(remaining) arrivals in the ith timeslot */
            servCount -= remain; /* remaining/new service power */
            this.lastproc = i;
            currArrTs++;
            currInner = 0;
            localList.add(remain);
            
            if (debug) {
               System.out.println("  Service is completed for timeslot = " + i + 
                                  " with remaining service power = " + servCount + ". CONTINUE....");
            }
            /* continue with the next timeslot */
         }
      }
      
      this.pArrivalTimeslot.add(currArrTs);
      this.pInnerPosition.add(currInner);
      this.pCounts.add(localList);
      
      if (debug) {
         System.out.println(" Processing info:" + " S_" + n + "=" + printProcessedCounts(n));
         System.out.println("  Next slot info: u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
         System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
      }
      return localList;
   }
   
   public void doFluidFlow() {
      int n = 0; // time interval
      while (n<totalTime) {
         if (this.debug) {
            System.out.println("n = " + n + " ------------------------------------------------------------------");
         }
         if (n == 10) {
            System.out.println("n = " + n);
         }
         int queueSize = getQueueSize(n);
         if (this.debug) {
            System.out.println("current queue-size[" + n + "]=" + queueSize);
         }
         
         int arrCount  = getArrivalCount(n);
         if (this.debug) {
            System.out.println("lambda[" + n + "]=" + arrCount);
         }
         this.totalArrivals += arrCount;
         
         int srvCount  = getServiceCount(n);
         if (this.debug) {
            System.out.println("mu[" + n + "]=" + srvCount);
         }
         int newQSize = queueSize + arrCount - srvCount;
         if (newQSize < 0) {
            this.serverIdleCount++;
            this.totalRemainingPower += (-1)*newQSize;
            this.totalServed += srvCount + newQSize;
            newQSize = 0;
         }
         else {
            this.totalServed += srvCount;
         }
                  
         if (this.debug) {
            System.out.println("primary queue-size[" + (n+1) + "]=" + newQSize);
         }
         queueSizes.add(newQSize);
         
         int removeCount = 0;
         ArrayList<Integer> procData = determineProcessedArrivalSlots(n);
         if (debug) {
            System.out.println("<><><><><><><><><><><><><><><><><><><><><><><><>");
            System.out.println("----Processed timeslot info: " + printProcessedCounts(n));
            System.out.println("----next arrival time slot = " + this.pArrivalTimeslot.get(n+1));
            System.out.println("----next inner position = " + this.pInnerPosition.get(n+1));
            System.out.println("<><><><><><><><><><><><><><><><><><><><><><><><>");
         }
         int procTS = this.pArrivalTimeslot.get(n);
         for (int i=procTS; i<procTS+procData.size(); i++) {
            for (int j=0; j<6; j++) {
               if ((n >= nt1) && (rtsizes[j].get(i) > 0)) {
                  int proc = rtsizes[j].get(i);
                  int servedCount = procData.get((i-procTS));
                  if (servedCount < proc) {
                     proc = servedCount;
                  }
                  removeCount += proc;
               }
            }
         }
         if (this.debug) {
            System.out.println("Newly Estimated removed count = " + removeCount);
         }
         this.totalRemovals += removeCount;
         
         int retCount  = getRetransCounts(n);
         if (this.debug) {
            System.out.println("r[" + n + "]=" + retCount);
         }
         this.totalRetransmissions += retCount;

         int calcRetransQueue = (this.secQueueSizes.get(n) - removeCount);
         if (calcRetransQueue < 0) {
            calcRetransQueue = 0;
         }
         calcRetransQueue += retCount;
         
         if (this.debug) {
            System.out.println("Calculated rt-queue= " + calcRetransQueue);
         }
         
         if ((calcRetransQueue > 0) && (newQSize == 0)) {
            System.err.println("ERROR: Unexpected situation  --> " + n);
         }
         
         this.secQueueSizes.add(calcRetransQueue);
         if (this.debug) {
            System.out.println("secondary queue-size[" + (n+1) + "]=" + calcRetransQueue);
         }
         if (debug) {
            System.out.println("Processing delay = " + (n-this.lastproc));
         }
         this.estimatedDelay.add(n-this.lastproc);
         
         int cease = checkForCeasedArrivals(n);
         if (debug) {
            System.out.println("ceased[" + n + "]=" + cease);
         }
         this.totalCeasedAttempts += cease;
         n++;
      }
      System.out.println("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
      System.out.println("Total arrivals = " + this.totalArrivals);
      System.out.println("Total served = " + this.totalServed);
      System.out.println("Total redundant power = " + this.totalRedundantPower);
      System.out.println("Total retransmissions = " + this.totalRetransmissions);
      System.out.println("Total removals = " + this.totalRemovals);
      System.out.println("Total server idle count = " + this.serverIdleCount);
      System.out.println("Total remaining power = " + this.totalRemainingPower);
      System.out.println("Total Ceased attempts = " + this.totalCeasedAttempts);
      System.out.println("Total Delay = " + this.totalDelay);
      System.out.println("Mean Delay = " + this.totalDelay/this.totalServed);
      for (int i=0; i<6; i++) {
         System.out.print("rt[" + (i+1) + "]=" + this.totalRetSeq[i] + " ");
      }
//      System.out.println(" ");
//      for (int i=0; i<this.retransRates.size(); i++) {
//         System.out.println("rtRate[" + + i + "]=" + this.retransRates.get(i));
//      }
      System.out.println("\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
   }
   
   public static void main(String[] args) {
      PrioRemoval sim = new PrioRemoval();
      sim.debug = true;
      sim.init();
      sim.doFluidFlow();
   }
}
