package infinitebuffer;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.LinkedBlockingQueue;

import base.Arrival;
import base.Deterministic_I;
import base.FluidFlowSim;


/**
 * This class implements fluid-flow simulation of conventional SIP
 * given by the fluid-flow analytical model define at the following
 * paper: 
 * Y. Hong, C. Huang, J. Yan, Impact of Retransmission Mechanism on 
 * SIP Overload: Stability Condition and Overload Control. 
 * JNW 7(1): 52-62, 2012.
 * 
 * Note that this implementation consists of a sanity check mechanism that
 * it creates retransmission timers for original and retransmitted arrivals
 * to verify the retransmission count consistency at a time-slot 
 * 
 * @author demiry
 *
 */
public class ConventionalSIPWithSanity_1 extends FluidFlowSim {

//   public static int t1n   = (int)t1/(int)intervalLength;
//   public static int t1_2n = (int)t1_2/(int)intervalLength;
//   public static int t1_3n = (int)t1_3/(int)intervalLength;
//   public static int t1_4n = (int)t1_4/(int)intervalLength;
//   public static int t1_5n = (int)t1_5/(int)intervalLength;
//   public static int t1_6n = (int)t1_6/(int)intervalLength;
   
   public LinkedBlockingQueue<Arrival> queue = new LinkedBlockingQueue<Arrival>();
   public int arrivalIndex = 0;
   public ArrayList<ArrayList<Arrival>> retransTimers = new ArrayList<ArrayList<Arrival>>(); 
   public ArrayList<Arrival> toBeCeased = new ArrayList<Arrival>();
   
   public ArrayList<Integer> pArrivalTimeslot = null; // u(n)
   public ArrayList<Integer> pInnerPosition   = null; // v(n)
   public ArrayList<ArrayList<Integer>> pCounts = null; // Sn(u,v)
   public ArrayList<ArrayList<Integer>> pdCounts = null; // Dn(u,v)
   
   public int lastProcessedSlot; /* actually processed from the queue */
   public int lastproc = 0; /* estimated one */
   
   public int calcProcForDelay;
   //public int procForDelay;
   public long calTotDelay = 0;
   
   //public ArrayList<Integer> estimatedDelay   = null;
   
   public void printQ() {
      if (debug) {
         Object[] obj = queue.toArray();
         for (int i = 0; i < obj.length; i++) {
            Arrival arr = (Arrival) obj[i];
            System.out.print(arr);
            System.out.print(", ");
            if ((i + 1) % 4 == 0) {
               System.out.println(" ");
            }
         }
         System.out.println(".. " + queue.size());
      }
   }
   
   /* determine and returns the next retransmission timer for an arrival */
   public int getNextRetransmitTime(Arrival arr, int n) {
      switch (arr.type) {
         case Arrival.NEW_ARRIVAL:
            arr.retTime = n+t1n;
            return Arrival.FIRST_RETRANS;
            
         case Arrival.FIRST_RETRANS:
            arr.retTime = n+t1_2n;
            return Arrival.SECOND_RETRANS;
            
         case Arrival.SECOND_RETRANS:
            arr.retTime = n+t1_3n;
            return Arrival.THIRD_RETRANS;
            
         case Arrival.THIRD_RETRANS:
            arr.retTime = n+t1_4n;
            return Arrival.FOURTH_RETRANS;
            
         case Arrival.FOURTH_RETRANS:
            arr.retTime = n+t1_5n;
            return Arrival.FIFTH_RETRANS;
            
         case Arrival.FIFTH_RETRANS:
            arr.retTime = n+t1_6n;
            return Arrival.SIXTH_RETRANS;
            
         case Arrival.SIXTH_RETRANS:
            return arr.type=-1;
      }
      return -1;
   }

   public void init() {
      arrivals = Deterministic_I.obtainArrivals((arrivalRate * intervalLength)/1000.0, totalTime);
      services = Deterministic_I.obtainServiceRates((serviceRate * intervalLength)/1000.0, totalTime);
      queueSizes = new ArrayList<Integer>(totalTime+1);
      queueSizes.add(queueSizeInitial);
      for (int i=0; i<6; i++) {
         retransTimers.add(new ArrayList<Arrival>());
      }
      for (int i=0; i<queueSizeInitial; i++) {
         Arrival arr = new Arrival(i, 0, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, 0);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
      }
      arrivals.set(0, 0);
      services.set(0, 0);
      retrans = new ArrayList<Integer>(totalTime+1);
      //retrans.add(0);
      
      this.pArrivalTimeslot = new ArrayList<Integer>(totalTime+1);
      this.pArrivalTimeslot.add(0);
      if (queueSizeInitial > 0) {
         this.pArrivalTimeslot.add(0); /* we need to start with initial-queue */
      }
      else {
         this.pArrivalTimeslot.add(1); /* trick way to start with arrival time slot 1 at processing time slot 1 */
      }
      this.pInnerPosition = new ArrayList<Integer>(totalTime+1);
      this.pInnerPosition.add(0);
      this.pInnerPosition.add(0); /* tricky way to start from 0 of starting arrival time slot */
      this.pCounts = new ArrayList<ArrayList<Integer>>();
      ArrayList<Integer> temp = new ArrayList<Integer>();
      temp.add(0);
      this.pCounts.add(temp);
      
      this.pdCounts = new ArrayList<ArrayList<Integer>>();
      temp = new ArrayList<Integer>();
      temp.add(0);
      this.pdCounts.add(temp);
      
      //this.estimatedDelay = new ArrayList<Integer>(totalTime+1);
   }
   
   /* returns arrival count at time slot n. And starts retransmission timer
    * for these arrivals.   
    */
   public int getArrivalCount(int n) {
      int dcount = arrivals.get(n);
      int count = (int)dcount;
      if (debug) {
         System.out.print("ARRIVALS[" + n + "]: ");
      }
      for (int i=0; i<count; i++) {
         Arrival arr = new Arrival(arrivalIndex, n, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, n);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
         if (debug) {
           //System.out.print(arr.id + ":" + arr.type + ":" + arr.retTime + ", ");
           System.out.print(arr + ", ");
         }
      }
      if (debug) {
         System.out.println("\n ...");
      }
      return dcount;
   }
   
   public  void stopRetransmitTimer(Arrival arr) {
      for (int i=0; i<retransTimers.size(); i++) {
         ArrayList<Arrival> rta = retransTimers.get(i);
         boolean res = rta.remove(arr);
         if (debug) {
            if (res) {
               System.out.print("+");
            }
            else {
               System.out.print("-");
            }
         }
      }
   }
   
   public void removeFromCeasableList(Arrival arr) {
      if (this.toBeCeased.remove(arr)) {
         if (debug) {
            System.out.print("Remove from TO_BE_CEASED: " + arr);
         }
      }
   }
   
   public int getServiceCount(int n) {
      int dsrvcount = services.get(n);
      int srvcount = (int)dsrvcount;
      if (debug) {
         System.out.print("SERVICE: ");
      }
      for (int i=0; i<srvcount; i++) {
         Arrival arr = queue.poll();
         if (arr != null) {
            if (debug) {
               //System.out.print(arr.id + ":" + arr.type + ":" + arr.retTime);// + ", ");
               System.out.print(arr);// + ", ");
            }
            this.lastProcessedSlot = arr.n;
            stopRetransmitTimer(arr);
            removeFromCeasableList(arr);
            if (arr.type <= 0) {
               /* new arrival */
               this.calcProcForDelay++;
               this.calTotDelay += n-arr.n;
            }
            if (debug) {
               System.out.print(", ");
            }
         }
         else {
            if (debug) {
               System.out.print("!");
            }
         }
      }
      if (debug) {
         System.out.println("\n ...");
      }
      return dsrvcount;
   }
   
   /* Calculates Eq. (3) */
   public int calcInternalRetrans(int n, int t1Count) {
      int val1 = 0;
      int val2 = 0;
      if ((n-t1Count) >= 0) {
         for (int i=0; i<=t1Count; i++) {
            int index = n-t1Count + i;
            val1 += services.get(index);
         }
         if ((n-t1Count) == 0) {
            val1 = queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = queueSizes.get(n-t1Count); // Assume arrivals at time 0
         }
         else {
            val1 = arrivals.get(n-t1Count) +  queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = arrivals.get(n-t1Count);
         }
      }
      
      return (val1 > val2) ? val2 : val1;
   }
   
   protected int checkForCeasedArrivals(int n) {
      int calcCeased = 0;
      calcCeased = calcInternalRetrans(n, nt1_6+t1n);
      
      int ceaseCount = 0;
      if (!this.toBeCeased.isEmpty()) {
//         return ceaseCount;  
//      }
         if (debug) {
            System.out.print("\r\nCEASED ARRIVALS: ");
         }
         Iterator<Arrival> lit = this.toBeCeased.iterator();
         while (lit.hasNext()) {
            Arrival arr = lit.next();
            if (arr.retTime > n) {
               break;
            }
            if (arr.retTime == n) {
               if (debug) {
                  System.out.print(arr);
               }
               lit.remove();
               if (debug) {
                  System.out.print(", ");
               }
               ceaseCount++;
            }
         }
         if (debug) {
            System.out.println("  CEASE TOTAL=" + ceaseCount);
//            if (ceaseCount > 0) {
//               System.out.println("  ceased=" + ceaseCount);
//            }
         }
      }
      if (calcCeased != ceaseCount) {
         System.err.println("ERROR: Calculated ceased=" + calcCeased + " and actual=" + ceaseCount + " are different for n=" + n);
      }
      return ceaseCount;
   }

   public void checkForSanity(int index, int value, int n) {
      ArrayList<Arrival> rta = retransTimers.get(index-1);
      int count = 0;
      //int ceased = 0;
      if (debug) {
         System.out.print("RT-arrivals<" + index + ">: ");
      }
      while (!rta.isEmpty()) {
         Arrival arr = rta.get(0);
         if (arr == null) {
            break;
         }
         if (arr.retTime != n) {
            break;
         }
         arr = rta.remove(0);
         //Arrival newArr = new Arrival(arr.id, n, arr.type+1);
         Arrival newArr = new Arrival(arr, n);
         //newArr.type++;
         int rt = getNextRetransmitTime(newArr, n);
         if (rt > -1) {
            /* retTime has been updated by getNextRetransmitTime */
            //newArr.type++;
            //newArr.n = n;
            ArrayList<Arrival> rta1 = retransTimers.get(rt-1);
            rta1.add(newArr);
//            queue.offer(newArr);
//            if (debug) {
//               System.out.print(newArr.id + ":" + newArr.type + ":" + newArr.retTime + ", ");
//            }
//            count++;
         }
         else {
            //ceased++;
            /* Retransmissions are ceases, put into the list for 
             * arrivals that to give-up the call. After the 6th 
             * retransmission at time 31.5s, there will be a 500ms
             * to give up the call (because of 64*T1=32s) 
             */
            newArr.type = -1;
            //newArr.n = n;
            newArr.retTime = n+t1n;
            this.toBeCeased.add(newArr);
         }
         queue.offer(newArr);
         if (debug) {
            //System.out.print(newArr.id + ":" + newArr.type + ":" + newArr.retTime + ", ");
            System.out.print(newArr + ", ");
         }
         count++;
      }
      if (debug) {
         Object[] arrTable = queue.toArray();
         if (debug) {
            System.out.println("\n size=" + arrTable.length + " ...");
            //System.out.println(" ...");
         }
      }
      if (count != value) {
         //System.err.println("ERROR: Estimated retransmit count<" + value + "> and found count<" + count + "> are different!!!" + " ceased=" + ceased + " for n=" +n);
         System.err.println("ERROR: Estimated retransmit count<" + value + "> and found count<" + count + "> are different!!! for n=" +n);
      }
   }

   /* Eq (2) */
   public int getRetransCounts(int n) {
      /* r(n) += rj(n), j=1,..,6  Eq(2) 
       * rj(n) = min{ [arr(n-Tj)+q(n-Tj) - (+)srv(n-Tj+k), k=1,..,Tj; j=1,..,6]positive, arr(n-Tj) } Eq(3)
       */
      int r = 0;
      int r1 = calcInternalRetrans(n, nt1);
      checkForSanity(Arrival.FIRST_RETRANS, (int)r1, n);
      
      int r2 = calcInternalRetrans(n, nt1_2); 
      checkForSanity(Arrival.SECOND_RETRANS, (int)r2, n);
      
      int r3 = calcInternalRetrans(n, nt1_3);
      checkForSanity(Arrival.THIRD_RETRANS, (int)r3, n);

      int r4 = calcInternalRetrans(n, nt1_4);
      checkForSanity(Arrival.FOURTH_RETRANS, (int)r4, n);
      
      int r5 = calcInternalRetrans(n, nt1_5);
      checkForSanity(Arrival.FIFTH_RETRANS, (int)r5, n);
      
      int r6 = calcInternalRetrans(n, nt1_6);
      checkForSanity(Arrival.SIXTH_RETRANS, (int)r6, n);      
      
      this.totalRetSeq[0] += r1;
      this.totalRetSeq[1] += r2;
      this.totalRetSeq[2] += r3;
      this.totalRetSeq[3] += r4;
      this.totalRetSeq[4] += r5;
      this.totalRetSeq[5] += r6;
      
      r = r1 + r2 + r3 + r4 + r5 + r6;
      if (debug) {
         System.out.println("total_r[" + n + "]=" + r + " (r1=" + r1 + ", r2=" + r2 + ", r3=" + r3 + ", r4=" + r4 + ", r5=" + r5 + ", r6=" + r6 + ")");
      }
      return r;
   }
   
   public int getQueueSize(int n) {
      return queueSizes.get(n);
   }

   private String printProcessedCounts(int timeslot) {
      if ((this.pCounts == null) || (this.pCounts.isEmpty())) return "";
      
      StringBuilder sb = new StringBuilder();
      ArrayList<Integer> pcList = this.pCounts.get(timeslot);
      if ((pcList == null) || (pcList.isEmpty())) return "";
      sb.append('{');
      int k = 0;
      if (timeslot > 0) {
         k = this.pArrivalTimeslot.get(timeslot);
      }
      for (int i=0; i<pcList.size(); i++) {
         sb.append("s(" + (k+i) + ")=" + pcList.get(i) + ", ");
      }
      sb.append('}');
      return sb.toString();
   }
   
   public ArrayList<Integer> determineProcessedArrivalSlots(int n) {
      ArrayList<Integer> localList = new ArrayList<Integer>();
      ArrayList<Integer> dlocalList = new ArrayList<Integer>();
      
      if (n == 0) {
         //return localList;
         return dlocalList;
      }
      int servCount = this.services.get(n);
      
      int currArrTs = this.pArrivalTimeslot.get(n);
      int currInner = this.pInnerPosition.get(n);
      
      if (debug) {
         System.out.println("~~~~~~~~~[" + n + "]~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
         System.out.println("Beginning: servPower=" + servCount + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
      }

      if (currArrTs == 0) {
         if (this.queueSizeInitial > 0) {
            /* Processing on initial-queue */
            currInner += servCount;
            
            if (currInner < this.queueSizeInitial) {
               /* still processing on initial-queue */
               localList.add(servCount);
               dlocalList.add(servCount);
               this.pArrivalTimeslot.add(currArrTs);
               this.pInnerPosition.add(currInner);
               this.pCounts.add(localList);
               this.pdCounts.add(dlocalList);
               
               if (debug) {
                  System.out.println("Still on the initial queue: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " S_" + n + "=" + printProcessedCounts(n));
               }
               //return localList;
               return dlocalList;
            }
            else if (currInner == this.queueSizeInitial) {
               /* Just finished the initial queue. We should continue with
                * the actual requests starting from the first arrival time slot 
                * at next processing time slot
                */
               localList.add(servCount);
               dlocalList.add(servCount);
               this.pArrivalTimeslot.add(1);
               this.pInnerPosition.add(0);
               this.pCounts.add(localList);
               this.pdCounts.add(dlocalList);
               
               //return localList;
               return dlocalList;
            }
            else {
               /* the processing of the initial-queue is just completed */
               /* there is a need to determine the processing for the actual queue */
               /* determine the last processed part from the initial-queue and
                * let the remaining to the following code-block for processing 
                */
               localList.add(servCount - (currInner - this.queueSizeInitial));
               dlocalList.add(servCount - (currInner - this.queueSizeInitial));
               servCount = currInner - this.queueSizeInitial; 
               currArrTs = 1;
               currInner = 0;
               if (debug) {
                  System.out.println("Initial queue is just completed: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
               }
            }
         }
      }
      
      //int lastproc = 0;
      /* Running on out of initial queue */
      for (int i=currArrTs; i<=n; i++) {
         //int remain = this.arrivals.get(i) - currInner;
         int remain = this.arrivals.get(i) + this.retrans.get(i)- currInner;
         int origArr = this.arrivals.get(i) - currInner;
         /* the time slot may have also retransmitted messages. So, currentInner may be greater than original arrivals */
         /* The main assumption here is that original arrivals always enter the queue before retransmitted messages */ 
         if (origArr < 0) {
            origArr = 0;
         }
         if (debug) {
            System.out.println("Serving for timeslot:" + i + " servPower=" + servCount + " remaining=" + remain + 
                               " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " origArr=" + origArr);
         }
         if (origArr > 0) {
            if (servCount <= origArr) {
               dlocalList.add(servCount);
            }
            else {
               dlocalList.add(origArr);
            }
         }
         else {
            dlocalList.add(0);
         }
         if (servCount < remain) {
            /* no change on processing timeslot but inner position */
            this.lastproc = i;
            currArrTs = i;
            currInner += servCount;
            localList.add(servCount);
            if (debug) {
               System.out.println("Still on timeslot = " + i + ": u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner );
            }
            break;
         }
         else if (servCount == remain) {
            /* promate to the next timeslot and initial position */
            this.lastproc = i;
            currArrTs = i+1;
            currInner = 0;
            localList.add(servCount);
            
            if (debug) {
               System.out.println("Service is just completed for timeslot = " + i + ": u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
            }
            break;
         }
         else {
            /* service power is bigger then(remaining) arrivals in the ith timeslot */
            servCount -= remain; /* remaining/new service power */
            lastproc = i;
            currArrTs++;
            currInner = 0;
            localList.add(remain);
            
            if (debug) {
               System.out.println("  Service is completed for timeslot = " + i + 
                                  " with remaining service power = " + servCount + ". CONTINUE....");
            }
            /* continue with the next timeslot */
         }
      }
      
      this.pArrivalTimeslot.add(currArrTs);
      this.pInnerPosition.add(currInner);
      this.pCounts.add(localList);
      
      if (this.lastproc != this.lastProcessedSlot) {
         System.err.println("Processed-" +  this.lastProcessedSlot + " and expected-" +  lastproc + " timeslots are different");
      }
      //System.out.println("Processing delay = " + (n-lastproc));
      
      if (debug) {
         System.out.println(" Processing info:" + " S_" + n + "=" + printProcessedCounts(n));
         System.out.println("  Next slot info: u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
         System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
      }
      //return localList;
      return dlocalList;
   }

   public void doFluidFlow() {
      int n = 0; // time interval
      int PTOT = 0;
      int DELAYTOT = 0;
      while (n<totalTime) {
         if (debug) {
            System.out.println("n = " + n + " ------------------------------------------------------------------");
            //printQ();
         }
         if (n==121) {
            System.out.println("n = " + n);
         }
         this.calcProcForDelay = 0; /* initialize the control/sanity value */
         //int queueSize = getQueueSize(n-1);
         int queueSize = getQueueSize(n);
         //System.out.println("prevq[" + (n-1) + "]=" + queueSize);
         if (debug) {
            System.out.println("current-queue[" + (n) + "]=" + queueSize);
         }
         
         int arrCount  = getArrivalCount(n);
         if (debug) {
            System.out.println("lambda[" + n + "]=" + arrCount);
         }
         this.totalArrivals += arrCount;
         
         int srvCount  = getServiceCount(n);
         if (debug) {
            System.out.println("mu[" + n + "]=" + srvCount);
         }
         
         //ArrayList<Integer> procData = determineProcessedArrivalSlots(n);
         
         int retCount  = getRetransCounts(n);
         if (debug) {
            System.out.println("r[" + n + "]=" + retCount);
         }
         this.totalRetransmissions += retCount;
         this.retrans.add(retCount);
         
         ArrayList<Integer> procData = determineProcessedArrivalSlots(n);
         
         /* Eq. (1) of reference. 
          */
         int newQSize = queueSize + arrCount + retCount - srvCount;
         if (newQSize < 0) {
            this.serverIdleCount++;
            this.totalRemainingPower += (-1)*newQSize;
            this.totalServed += srvCount + newQSize;
            newQSize = 0;
         }
         else {
            this.totalServed += srvCount;
         }
         
         if (debug) {
            System.out.println("queue-size[" + (n+1) + "] = " + newQSize);
         }
         
         int qsize = this.queue.size();
         if (qsize != newQSize) {
            System.err.println("ERROR: Actual queue size=" + qsize + " and calculated=" + newQSize + " are different for n=" +n);
         }
         //queueSizes.add(newQSize);
         queueSizes.add(qsize);
         
         int k = 0;
         int ptot = 0;
         int diffTot = 0;
         //int delayTot = 0;
         System.out.println("Processing delay = " + (n-this.lastproc));
         System.out.println("u[n]=" + this.pArrivalTimeslot.get(n) + " u[n+1]=" + this.pArrivalTimeslot.get(n+1));
         k = this.pArrivalTimeslot.get(n);
         for (int i=0; i<procData.size(); i++) {
            System.out.println("s[" + (k+i) + "]=" + procData.get(i));
            ptot += procData.get(i);
            diffTot += n-(k+i);
            this.totalDelay += (n-(k+i))*procData.get(i);
            this.actualServing += procData.get(i);
            this.meanDelay = (double)this.totalDelay/this.actualServing;
            System.out.println("Total delay=" + this.totalDelay + " server=" + this.actualServing + " mean-delay=" + this.meanDelay);
            //delayTot += (n-(k+i))*procData.get(i);
            //PTOT += ptot;
            //DELAYTOT += delayTot;
         }
         if (ptot != this.calcProcForDelay) {
            System.err.println("ERROR: Calculated delay count:" + this.calcProcForDelay + " and estimated one:" + ptot + " are different");
         }
         int diff = 0;
         if (!procData.isEmpty()) {
            diff = diffTot/procData.size();
         }
         if (debug) {
            System.out.println("Processing delay = " + (n-this.lastproc));
            System.out.println("Mean Processing delay = " + diff);
            System.out.println("XXXTotal delay=" + this.totalDelay + " server=" + this.actualServing + " mean-delay=" + this.meanDelay);
            //System.out.println("Delay Total = " + delayTot + " for n=" + n + " PTOT=" + PTOT + " DELAYTOT=" + DELAYTOT + " MEAN=" + (double)DELAYTOT/PTOT);
         }
         //this.estimatedDelay.add(n-this.lastproc);
         //this.estimatedDelay.add(diff);
         //this.totalDelay += (n-this.lastproc);
//         this.totalDelay += diff*ptot;
         
         int cease = checkForCeasedArrivals(n);
         if (debug) {
            System.out.println("ceased[" + n + "]=" + cease);
         }
         this.totalCeasedAttempts += cease;
         
         n++;
      }
      System.out.println("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
      System.out.println("Total arrivals = " + this.totalArrivals);
      System.out.println("Total served = " + this.totalServed);
      System.out.println("Total retransmissions = " + this.totalRetransmissions);
      System.out.println("Total server idle count = " + this.serverIdleCount);
      System.out.println("Total remaining power = " + this.totalRemainingPower);
      System.out.println("Total Ceased attempts = " + this.totalCeasedAttempts);
      long inms = this.totalDelay*intervalLength;
      double inSec = inms/1000;
      System.out.println("Total Delay = " + this.totalDelay + " in ms = " + inms + " in sec = " + inSec);
      System.out.println("Mean Delay = " + inSec/this.totalArrivals);
      inms = this.calTotDelay*intervalLength;
      inSec = inms/1000;
      System.out.println("Calc Mean Delay = " + inSec/this.totalArrivals);
//      long estTot = 0;
//      for (int i=0; i<this.estimatedDelay.size(); i++) {
//         estTot += this.estimatedDelay.get(i);
//      }
//      System.out.println("Estimated delay in total = " + estTot + " with mean(sec) = " + 
//                         (((double)estTot/this.estimatedDelay.size())*intervalLength/1000)); 
      for (int i=0; i<6; i++) {
         System.out.print("rt[" + (i+1) + "]=" + this.totalRetSeq[i] + " ");
      }
      System.out.println("\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
   }
   
   public static void main(String[] args) {
      ConventionalSIPWithSanity sim = new ConventionalSIPWithSanity(); 
      sim.init();
      sim.doFluidFlow();
   }
}
