package infinitebuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.LinkedBlockingQueue;

import base.Arrival;
import base.Deterministic_I;
import base.FluidFlowSim;


/**
 * This class implements fluid-flow simulation of conventional SIP
 * given by the fluid-flow analytical model define at the following
 * paper: 
 * Y. Hong, C. Huang, J. Yan, Impact of Retransmission Mechanism on 
 * SIP Overload: Stability Condition and Overload Control. 
 * JNW 7(1): 52-62, 2012.
 * 
 * Note that this implementation consists of a sanity check mechanism that
 * it creates retransmission timers for original and retransmitted arrivals
 * to verify the retransmission count consistency at a time-slot 
 * 
 * @author demiry
 *
 */
public class ConventionalSIPWithSanity extends FluidFlowSim {

//   public static int t1n   = (int)t1/(int)intervalLength;
//   public static int t1_2n = (int)t1_2/(int)intervalLength;
//   public static int t1_3n = (int)t1_3/(int)intervalLength;
//   public static int t1_4n = (int)t1_4/(int)intervalLength;
//   public static int t1_5n = (int)t1_5/(int)intervalLength;
//   public static int t1_6n = (int)t1_6/(int)intervalLength;
   
   public LinkedBlockingQueue<Arrival> queue = new LinkedBlockingQueue<Arrival>();
   public int arrivalIndex = 0;
   public ArrayList<ArrayList<Arrival>> retransTimers = new ArrayList<ArrayList<Arrival>>(); 
   public ArrayList<Arrival> toBeCeased = new ArrayList<Arrival>();
   
   public void printQ() {
      if (debug) {
         Object[] obj = queue.toArray();
         for (int i = 0; i < obj.length; i++) {
            Arrival arr = (Arrival) obj[i];
            System.out.print(arr);
            System.out.print(", ");
            if ((i + 1) % 4 == 0) {
               System.out.println(" ");
            }
         }
         System.out.println(".. " + queue.size());
      }
   }
   
   /* determine and returns the next retransmission timer for an arrival */
   public int getNextRetransmitTime(Arrival arr, int n) {
      switch (arr.type) {
         case Arrival.NEW_ARRIVAL:
            arr.retTime = n+t1n;
            return Arrival.FIRST_RETRANS;
            
         case Arrival.FIRST_RETRANS:
            arr.retTime = n+t1_2n;
            return Arrival.SECOND_RETRANS;
            
         case Arrival.SECOND_RETRANS:
            arr.retTime = n+t1_3n;
            return Arrival.THIRD_RETRANS;
            
         case Arrival.THIRD_RETRANS:
            arr.retTime = n+t1_4n;
            return Arrival.FOURTH_RETRANS;
            
         case Arrival.FOURTH_RETRANS:
            arr.retTime = n+t1_5n;
            return Arrival.FIFTH_RETRANS;
            
         case Arrival.FIFTH_RETRANS:
            arr.retTime = n+t1_6n;
            return Arrival.SIXTH_RETRANS;
            
         case Arrival.SIXTH_RETRANS:
            return arr.type=-1;
      }
      return -1;
   }

   public void init() {
      arrivals = Deterministic_I.obtainArrivals((arrivalRate * intervalLength)/1000.0, totalTime);
      services = Deterministic_I.obtainServiceRates((serviceRate * intervalLength)/1000.0, totalTime);
      queueSizes = new ArrayList<Integer>(totalTime+1);
      queueSizes.add(queueSizeInitial);
      for (int i=0; i<6; i++) {
         retransTimers.add(new ArrayList<Arrival>());
      }
      for (int i=0; i<queueSizeInitial; i++) {
         Arrival arr = new Arrival(i, 0, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, 0);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
      }
      arrivals.set(0, 0);
      services.set(0, 0);
   }
   
   /* returns arrival count at time slot n. And starts retransmission timer
    * for these arrivals.   
    */
   public int getArrivalCount(int n) {
      int dcount = arrivals.get(n);
      int count = (int)dcount;
      if (debug) {
         System.out.print("ARRIVALS[" + n + "]: ");
      }
      for (int i=0; i<count; i++) {
         Arrival arr = new Arrival(arrivalIndex, n, 0);
         queue.offer(arr);
         arrivalIndex++;
         int rt = getNextRetransmitTime(arr, n);
         ArrayList<Arrival> rta = retransTimers.get(rt-1);
         rta.add(arr);
         if (debug) {
           //System.out.print(arr.id + ":" + arr.type + ":" + arr.retTime + ", ");
           System.out.print(arr + ", ");
         }
      }
      if (debug) {
         System.out.println("\n ...");
      }
      return dcount;
   }
   
   public  void stopRetransmitTimer(Arrival arr) {
      for (int i=0; i<retransTimers.size(); i++) {
         ArrayList<Arrival> rta = retransTimers.get(i);
         boolean res = rta.remove(arr);
         if (debug) {
            if (res) {
               System.out.print("+");
            }
            else {
               System.out.print("-");
            }
         }
      }
   }
   
   public void removeFromCeasableList(Arrival arr) {
      if (this.toBeCeased.remove(arr)) {
         if (debug) {
            System.out.print("Remove from TO_BE_CEASED: " + arr);
         }
      }
   }
   
   public int getServiceCount(int n) {
      int dsrvcount = services.get(n);
      int srvcount = (int)dsrvcount;
      if (debug) {
         System.out.print("SERVICE: ");
      }
      for (int i=0; i<srvcount; i++) {
         Arrival arr = queue.poll();
         if (arr != null) {
            if (debug) {
               //System.out.print(arr.id + ":" + arr.type + ":" + arr.retTime);// + ", ");
               System.out.print(arr);// + ", ");
            }
            stopRetransmitTimer(arr);
            removeFromCeasableList(arr);
            if (debug) {
               System.out.print(", ");
            }
         }
         else {
            if (debug) {
               System.out.print("!");
            }
         }
      }
      if (debug) {
         System.out.println("\n ...");
      }
      return dsrvcount;
   }
   
   /* Calculates Eq. (3) */
   public int calcInternalRetrans(int n, int t1Count) {
      int val1 = 0;
      int val2 = 0;
      if ((n-t1Count) >= 0) {
         for (int i=0; i<=t1Count; i++) {
            int index = n-t1Count + i;
            val1 += services.get(index);
         }
         if ((n-t1Count) == 0) {
            val1 = queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = queueSizes.get(n-t1Count); // Assume arrivals at time 0
         }
         else {
            val1 = arrivals.get(n-t1Count) +  queueSizes.get(n-t1Count) - val1;
            if (val1 < 0) {
               val1 = 0;
            }
            val2 = arrivals.get(n-t1Count);
         }
      }
      
      return (val1 > val2) ? val2 : val1;
   }
   
   protected int checkForCeasedArrivals(int n) {
      int calcCeased = 0;
      calcCeased = calcInternalRetrans(n, nt1_6+t1n);
      
      int ceaseCount = 0;
      if (!this.toBeCeased.isEmpty()) {
//         return ceaseCount;  
//      }
         if (debug) {
            System.out.print("\r\nCEASED ARRIVALS: ");
         }
         Iterator<Arrival> lit = this.toBeCeased.iterator();
         while (lit.hasNext()) {
            Arrival arr = lit.next();
            if (arr.retTime > n) {
               break;
            }
            if (arr.retTime == n) {
               if (debug) {
                  System.out.print(arr);
               }
               lit.remove();
               if (debug) {
                  System.out.print(", ");
               }
               ceaseCount++;
            }
         }
         if (debug) {
            System.out.println("  CEASE TOTAL=" + ceaseCount);
//            if (ceaseCount > 0) {
//               System.out.println("  ceased=" + ceaseCount);
//            }
         }
      }
      if (calcCeased != ceaseCount) {
         System.err.println("ERROR: Calculated ceased=" + calcCeased + " and actual=" + ceaseCount + " are different for n=" + n);
      }
      return ceaseCount;
   }

   public void checkForSanity(int index, int value, int n) {
      ArrayList<Arrival> rta = retransTimers.get(index-1);
      int count = 0;
      //int ceased = 0;
      if (debug) {
         System.out.print("RT-arrivals<" + index + ">: ");
      }
      while (!rta.isEmpty()) {
         Arrival arr = rta.get(0);
         if (arr == null) {
            break;
         }
         if (arr.retTime != n) {
            break;
         }
         arr = rta.remove(0);
         //Arrival newArr = new Arrival(arr.id, n, arr.type+1);
         Arrival newArr = new Arrival(arr, n);
         //newArr.type++;
         int rt = getNextRetransmitTime(newArr, n);
         if (rt > -1) {
            /* retTime has been updated by getNextRetransmitTime */
            //newArr.type++;
            //newArr.n = n;
            ArrayList<Arrival> rta1 = retransTimers.get(rt-1);
            rta1.add(newArr);
//            queue.offer(newArr);
//            if (debug) {
//               System.out.print(newArr.id + ":" + newArr.type + ":" + newArr.retTime + ", ");
//            }
//            count++;
         }
         else {
            //ceased++;
            /* Retransmissions are ceases, put into the list for 
             * arrivals that to give-up the call. After the 6th 
             * retransmission at time 31.5s, there will be a 500ms
             * to give up the call (because of 64*T1=32s) 
             */
            newArr.type = -1;
            //newArr.n = n;
            newArr.retTime = n+t1n;
            this.toBeCeased.add(newArr);
         }
         queue.offer(newArr);
         if (debug) {
            //System.out.print(newArr.id + ":" + newArr.type + ":" + newArr.retTime + ", ");
            System.out.print(newArr + ", ");
         }
         count++;
      }
      if (debug) {
         Object[] arrTable = queue.toArray();
         if (debug) {
            System.out.println("\n size=" + arrTable.length + " ...");
            //System.out.println(" ...");
         }
      }
      if (count != value) {
         //System.err.println("ERROR: Estimated retransmit count<" + value + "> and found count<" + count + "> are different!!!" + " ceased=" + ceased + " for n=" +n);
         System.err.println("ERROR: Estimated retransmit count<" + value + "> and found count<" + count + "> are different!!! for n=" +n);
      }
   }

   /* Eq (2) */
   public int getRetransCounts(int n) {
      /* r(n) += rj(n), j=1,..,6  Eq(2) 
       * rj(n) = min{ [arr(n-Tj)+q(n-Tj) - (+)srv(n-Tj+k), k=1,..,Tj; j=1,..,6]positive, arr(n-Tj) } Eq(3)
       */
      int r = 0;
      int r1 = calcInternalRetrans(n, nt1);
      checkForSanity(Arrival.FIRST_RETRANS, (int)r1, n);
      
      int r2 = calcInternalRetrans(n, nt1_2); 
      checkForSanity(Arrival.SECOND_RETRANS, (int)r2, n);
      
      int r3 = calcInternalRetrans(n, nt1_3);
      checkForSanity(Arrival.THIRD_RETRANS, (int)r3, n);

      int r4 = calcInternalRetrans(n, nt1_4);
      checkForSanity(Arrival.FOURTH_RETRANS, (int)r4, n);
      
      int r5 = calcInternalRetrans(n, nt1_5);
      checkForSanity(Arrival.FIFTH_RETRANS, (int)r5, n);
      
      int r6 = calcInternalRetrans(n, nt1_6);
      checkForSanity(Arrival.SIXTH_RETRANS, (int)r6, n);      
      
      this.totalRetSeq[0] += r1;
      this.totalRetSeq[1] += r2;
      this.totalRetSeq[2] += r3;
      this.totalRetSeq[3] += r4;
      this.totalRetSeq[4] += r5;
      this.totalRetSeq[5] += r6;
      
      r = r1 + r2 + r3 + r4 + r5 + r6;
      if (debug) {
         System.out.println("total_r[" + n + "]=" + r + " (r1=" + r1 + ", r2=" + r2 + ", r3=" + r3 + ", r4=" + r4 + ", r5=" + r5 + ", r6=" + r6 + ")");
      }
      return r;
   }
   
   public int getQueueSize(int n) {
      return queueSizes.get(n);
   }

   public void doFluidFlow() {
      int n = 0; // time interval
      while (n<totalTime) {
         if (debug) {
            System.out.println("n = " + n + " ------------------------------------------------------------------");
            //printQ();
         }
//         if (n==23726) {
//            System.out.println("n = " + n);
//         }
         //int queueSize = getQueueSize(n-1);
         int queueSize = getQueueSize(n);
         //System.out.println("prevq[" + (n-1) + "]=" + queueSize);
         if (debug) {
            System.out.println("current-queue[" + (n) + "]=" + queueSize);
         }
         
         int arrCount  = getArrivalCount(n);
         if (debug) {
            System.out.println("lambda[" + n + "]=" + arrCount);
         }
         this.totalArrivals += arrCount;
         
         int srvCount  = getServiceCount(n);
         if (debug) {
            System.out.println("mu[" + n + "]=" + srvCount);
         }
         
         int retCount  = getRetransCounts(n);
         if (debug) {
            System.out.println("r[" + n + "]=" + retCount);
         }
         this.totalRetransmissions += retCount;
         
         /* Eq. (1) of reference. 
          */
         int newQSize = queueSize + arrCount + retCount - srvCount;
         if (newQSize < 0) {
            this.serverIdleCount++;
            this.totalRemainingPower += (-1)*newQSize;
            this.totalServed += srvCount + newQSize;
            newQSize = 0;
         }
         else {
            this.totalServed += srvCount;
         }
         
         if (debug) {
            System.out.println("queue-size[" + (n+1) + "] = " + newQSize);
         }
         
         int qsize = this.queue.size();
         if (qsize != newQSize) {
            System.err.println("ERROR: Actual queue size=" + qsize + " and calculated=" + newQSize + " are different for n=" +n);
         }
         //queueSizes.add(newQSize);
         queueSizes.add(qsize);
         
         int cease = checkForCeasedArrivals(n);
         if (debug) {
            System.out.println("ceased[" + n + "]=" + cease);
         }
         this.totalCeasedAttempts += cease;
         
         n++;
      }
      System.out.println("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
      System.out.println("Total arrivals = " + this.totalArrivals);
      System.out.println("Total served = " + this.totalServed);
      System.out.println("Total retransmissions = " + this.totalRetransmissions);
      System.out.println("Total server idle count = " + this.serverIdleCount);
      System.out.println("Total remaining power = " + this.totalRemainingPower);
      System.out.println("Total Ceased attempts = " + this.totalCeasedAttempts);
      for (int i=0; i<6; i++) {
         System.out.print("rt[" + (i+1) + "]=" + this.totalRetSeq[i] + " ");
      }
      System.out.println("\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
   }
   
   public static void main(String[] args) {
      ConventionalSIPWithSanity sim = new ConventionalSIPWithSanity(); 
      sim.init();
      sim.doFluidFlow();
   }
}
