package infinitebuffer;
import java.util.ArrayList;
import java.util.Iterator;

import base.Arrival;
import base.Deterministic_I;
import base.FluidFlowSim;


/**
 * This class implements fluid-flow simulation of conventional SIP
 * given by the fluid-flow analytical model defined in the following paper: 
 * Y. Hong, C. Huang, J. Yan, Impact of Retransmission Mechanism on 
 *                             SIP Overload: Stability Condition and Overload 
 *                             Control.  JNW 7(1): 52-62, 2012.
 * @author demiry
 *
 */
public class ConvSIP  extends FluidFlowSim {

   public ArrayList<Integer> pArrivalTimeslot = null; // u(n)
   public ArrayList<Integer> pInnerPosition   = null; // v(n)
   public ArrayList<ArrayList<Integer>> pCounts = null; // Sn(u,v)
   public ArrayList<ArrayList<Integer>> pdCounts = null; // Dn(u,v)
   
   public int lastproc = 0; /* estimated one */
   
   public ArrayList<Integer> estimatedDelay   = null;
   
   public void init() {
      arrivals = Deterministic_I.obtainArrivals((int) ((arrivalRate * intervalLength)/1000.0), totalTime);
      services = Deterministic_I.obtainServiceRates((int) ((serviceRate * intervalLength)/1000.0), totalTime);
      queueSizes = new ArrayList<Integer>(totalTime+1);
      queueSizes.add(queueSizeInitial);
      arrivals.add(0, 0);
      services.add(0, 0);
      
      retrans = new ArrayList<Integer>(totalTime+1);
      this.pArrivalTimeslot = new ArrayList<Integer>(totalTime+1);
      this.pArrivalTimeslot.add(0);
      if (queueSizeInitial > 0) {
         this.pArrivalTimeslot.add(0); /* we need to start with initial-queue */
      }
      else {
         this.pArrivalTimeslot.add(1); /* trick way to start with arrival time slot 1 at processing time slot 1 */
      }
      this.pInnerPosition = new ArrayList<Integer>(totalTime+1);
      this.pInnerPosition.add(0);
      this.pInnerPosition.add(0); /* tricky way to start from 0 of starting arrival time slot */
      this.pCounts = new ArrayList<ArrayList<Integer>>();
      ArrayList<Integer> temp = new ArrayList<Integer>();
      temp.add(0);
      this.pCounts.add(temp);
      
      this.pdCounts = new ArrayList<ArrayList<Integer>>();
      temp = new ArrayList<Integer>();
      temp.add(0);
      this.pdCounts.add(temp);
      
      this.estimatedDelay = new ArrayList<Integer>(totalTime+1);
   }
   
   protected int checkForCeasedArrivals(int n) {
      int calcCeased = 0;
      calcCeased = calcInternalRetrans(n, nt1_6+t1n);
      
      return calcCeased;
   }
   
   private String printProcessedCounts(int timeslot) {
      if ((this.pCounts == null) || (this.pCounts.isEmpty())) return "";
      
      StringBuilder sb = new StringBuilder();
      ArrayList<Integer> pcList = this.pCounts.get(timeslot);
      if ((pcList == null) || (pcList.isEmpty())) return "";
      sb.append('{');
      int k = 0;
      if (timeslot > 0) {
         k = this.pArrivalTimeslot.get(timeslot);
      }
      for (int i=0; i<pcList.size(); i++) {
         sb.append("s(" + (k+i) + ")=" + pcList.get(i) + ", ");
      }
      sb.append('}');
      return sb.toString();
   }

   public ArrayList<Integer> determineProcessedArrivalSlots_1(int n) {
      ArrayList<Integer> localList = new ArrayList<Integer>();
      
      if (n == 0) {
         return localList;
      }
      int servCount = this.services.get(n);
      
      int currArrTs = this.pArrivalTimeslot.get(n);
      int currInner = this.pInnerPosition.get(n);
      
      if (debug) {
         System.out.println("~~~~~~~~~[" + n + "]~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
         System.out.println("Beginning: servPower=" + servCount + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
      }

      if (currArrTs == 0) {
         if (this.queueSizeInitial > 0) {
            /* Processing on initial-queue */
            currInner += servCount;
            
            if (currInner < this.queueSizeInitial) {
               /* still processing on initial-queue */
               localList.add(servCount);
               this.pArrivalTimeslot.add(currArrTs);
               this.pInnerPosition.add(currInner);
               this.pCounts.add(localList);
               
               if (debug) {
                  System.out.println("Still on the initial queue: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " S_" + n + "=" + printProcessedCounts(n));
               }
               return localList;
            }
            else if (currInner == this.queueSizeInitial) {
               /* Just finished the initial queue. We should continue with
                * the actual requests starting from the first arrival time slot 
                * at next processing time slot
                */
               localList.add(servCount);
               this.pArrivalTimeslot.add(1);
               this.pInnerPosition.add(0);
               this.pCounts.add(localList);
               
               return localList;
            }
            else {
               /* the processing of the initial-queue is just completed */
               /* there is a need to determine the processing for the actual queue */
               /* determine the last processed part from the initial-queue and
                * let the remaining to the following code-block for processing 
                */
               localList.add(servCount - (currInner - this.queueSizeInitial));
               servCount = currInner - this.queueSizeInitial; 
               currArrTs = 1;
               currInner = 0;
               if (debug) {
                  System.out.println("Initial queue is just completed: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
               }
            }
         }
      }
      
      //int lastproc = 0;
      /* Running on out of initial queue */
      for (int i=currArrTs; i<=n; i++) {
         //int remain = this.arrivals.get(i) - currInner;
         int remain = this.arrivals.get(i) + this.retrans.get(i)- currInner;
         if (debug) {
            System.out.println("Serving for timeslot:" + i + " servPower=" + servCount + " remaining=" + remain + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
         }
         if (servCount < remain) {
            /* no change on processing timeslot but inner position */
            this.lastproc = i;
            currArrTs = i;
            currInner += servCount;
            localList.add(servCount);
            if (debug) {
               System.out.println("Still on timeslot = " + i + ": u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner );
            }
            break;
         }
         else if (servCount == remain) {
            /* promate to the next timeslot and initial position */
            this.lastproc = i;
            currArrTs = i+1;
            currInner = 0;
            localList.add(servCount);
            
            if (debug) {
               System.out.println("Service is just completed for timeslot = " + i + ": u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
            }
            break;
         }
         else {
            /* service power is bigger then(remaining) arrivals in the ith timeslot */
            servCount -= remain; /* remaining/new service power */
            lastproc = i;
            currArrTs++;
            currInner = 0;
            localList.add(remain);
            
            if (debug) {
               System.out.println("  Service is completed for timeslot = " + i + 
                                  " with remaining service power = " + servCount + ". CONTINUE....");
            }
            /* continue with the next timeslot */
         }
      }
      
      this.pArrivalTimeslot.add(currArrTs);
      this.pInnerPosition.add(currInner);
      this.pCounts.add(localList);
      
      //System.out.println("Processing delay = " + (n-lastproc));
      
      if (debug) {
         System.out.println(" Processing info:" + " S_" + n + "=" + printProcessedCounts(n));
         System.out.println("  Next slot info: u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
         System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
      }
      return localList;
   }

   public ArrayList<Integer> determineProcessedArrivalSlots(int n) {
      ArrayList<Integer> localList = new ArrayList<Integer>();
      ArrayList<Integer> dlocalList = new ArrayList<Integer>();
      
      if (n == 0) {
         //return localList;
         return dlocalList;
      }
      int servCount = this.services.get(n);
      
      int currArrTs = this.pArrivalTimeslot.get(n);
      int currInner = this.pInnerPosition.get(n);
      
      if (debug) {
         System.out.println("~~~~~~~~~[" + n + "]~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
         System.out.println("Beginning: servPower=" + servCount + " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
      }

      if (currArrTs == 0) {
         if (this.queueSizeInitial > 0) {
            /* Processing on initial-queue */
            currInner += servCount;
            
            if (currInner < this.queueSizeInitial) {
               /* still processing on initial-queue */
               localList.add(servCount);
               dlocalList.add(servCount);
               this.pArrivalTimeslot.add(currArrTs);
               this.pInnerPosition.add(currInner);
               this.pCounts.add(localList);
               this.pdCounts.add(dlocalList);
               
               if (debug) {
                  System.out.println("Still on the initial queue: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " S_" + n + "=" + printProcessedCounts(n));
               }
               //return localList;
               return dlocalList;
            }
            else if (currInner == this.queueSizeInitial) {
               /* Just finished the initial queue. We should continue with
                * the actual requests starting from the first arrival time slot 
                * at next processing time slot
                */
               localList.add(servCount);
               dlocalList.add(servCount);
               this.pArrivalTimeslot.add(1);
               this.pInnerPosition.add(0);
               this.pCounts.add(localList);
               this.pdCounts.add(dlocalList);
               
               //return localList;
               return dlocalList;
            }
            else {
               /* the processing of the initial-queue is just completed */
               /* there is a need to determine the processing for the actual queue */
               /* determine the last processed part from the initial-queue and
                * let the remaining to the following code-block for processing 
                */
               localList.add(servCount - (currInner - this.queueSizeInitial));
               dlocalList.add(servCount - (currInner - this.queueSizeInitial));
               servCount = currInner - this.queueSizeInitial; 
               currArrTs = 1;
               currInner = 0;
               if (debug) {
                  System.out.println("Initial queue is just completed: u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner);
               }
            }
         }
      }
      
      //int lastproc = 0;
      /* Running on out of initial queue */
      for (int i=currArrTs; i<=n; i++) {
         //int remain = this.arrivals.get(i) - currInner;
         int remain = this.arrivals.get(i) + this.retrans.get(i)- currInner;
         int origArr = this.arrivals.get(i) - currInner;
         /* the time slot may have also retransmitted messages. So, currentInner may be greater than original arrivals */
         /* The main assumption here is that original arrivals always enter the queue before retransmitted messages */ 
         if (origArr < 0) {
            origArr = 0;
         }
         if (debug) {
            System.out.println("Serving for timeslot:" + i + " servPower=" + servCount + " remaining=" + remain + 
                               " u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner + " origArr=" + origArr);
         }
         if (origArr > 0) {
            if (servCount <= origArr) {
               dlocalList.add(servCount);
            }
            else {
               dlocalList.add(origArr);
            }
         }
         else {
            dlocalList.add(0);
         }
         if (servCount < remain) {
            /* no change on processing timeslot but inner position */
            this.lastproc = i;
            currArrTs = i;
            currInner += servCount;
            localList.add(servCount);
            if (debug) {
               System.out.println("Still on timeslot = " + i + ": u[" + n + "]=" + currArrTs + " v[" + n + "]=" + currInner );
            }
            break;
         }
         else if (servCount == remain) {
            /* promate to the next timeslot and initial position */
            this.lastproc = i;
            currArrTs = i+1;
            currInner = 0;
            localList.add(servCount);
            
            if (debug) {
               System.out.println("Service is just completed for timeslot = " + i + ": u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
            }
            break;
         }
         else {
            /* service power is bigger then(remaining) arrivals in the ith timeslot */
            servCount -= remain; /* remaining/new service power */
            lastproc = i;
            currArrTs++;
            currInner = 0;
            localList.add(remain);
            
            if (debug) {
               System.out.println("  Service is completed for timeslot = " + i + 
                                  " with remaining service power = " + servCount + ". CONTINUE....");
            }
            /* continue with the next timeslot */
         }
      }
      
      this.pArrivalTimeslot.add(currArrTs);
      this.pInnerPosition.add(currInner);
      this.pCounts.add(localList);
      
//      if (this.lastproc != this.lastProcessedSlot) {
//         System.err.println("Processed-" +  this.lastProcessedSlot + " and expected-" +  lastproc + " timeslots are different");
//      }
      //System.out.println("Processing delay = " + (n-lastproc));
      
      if (debug) {
         System.out.println(" Processing info:" + " S_" + n + "=" + printProcessedCounts(n));
         System.out.println("  Next slot info: u[" + (n+1) + "]=" + currArrTs + " v[" + (n+1) + "]=" + currInner);
         System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
      }
      //return localList;
      return dlocalList;
   }

   public void doFluidFlow() {
      int n = 0; // time interval
      while (n<totalTime) {
         if (debug) {
            System.out.println("n = " + n + " ------------------------------------------------------------------");
         }
         
         int queueSize = getQueueSize(n);
         //int queueSize = getQueueSize(n-1);
         if (debug) {
            System.out.println("current-queue[" + (n) + "]=" + queueSize);
         }
         
         int arrCount  = getArrivalCount(n);
         if (debug) {
            System.out.println("lambda[" + n + "]=" + arrCount);
         }
         this.totalArrivals += arrCount;
         
         int srvCount  = getServiceCount(n);
         if (debug) {
            System.out.println("mu[" + n + "]=" + srvCount);
         }

         int retCount  = getRetransCounts(n);
         if (debug) {
            System.out.println("r[" + n + "]=" + retCount);
         }
         this.totalRetransmissions += retCount;
         this.retrans.add(retCount);
         
         ArrayList<Integer> procData = determineProcessedArrivalSlots(n);
         
         /* Eq. (1) of reference. 
          */
         int newQSize = queueSize + arrCount + retCount - srvCount;
         if (newQSize < 0) {
            this.serverIdleCount++;
            this.totalRemainingPower += (-1)*newQSize;
            this.totalServed += srvCount + newQSize;
            newQSize = 0;
         }
         else {
            this.totalServed += srvCount;
         }
         
         if (debug) {
            System.out.println("queue-size[" + (n+1) + "] = " + newQSize);
         }
         queueSizes.add(newQSize);
         if (debug) {
            System.out.println("Processing delay = " + (n-this.lastproc));
         }
         //this.estimatedDelay.add(n-this.lastproc);
         //this.totalDelay += (n-this.lastproc);
         int k = 0;
         int ptot = 0;
         int diffTot = 0;
         System.out.println("Processing delay = " + (n-this.lastproc));
         System.out.println("u[n]=" + this.pArrivalTimeslot.get(n) + " u[n+1]=" + this.pArrivalTimeslot.get(n+1));
         k = this.pArrivalTimeslot.get(n);
         for (int i=0; i<procData.size(); i++) {
            System.out.println("s[" + (k+i) + "]=" + procData.get(i));
            ptot += procData.get(i);
            diffTot += n-(k+i);
         }
//         if (ptot != this.calcProcForDelay) {
//            System.err.println("ERROR: Calculated delay count:" + this.calcProcForDelay + " and estimated one:" + ptot + " are different");
//         }
         int diff = 0;
         if (!procData.isEmpty()) {
            diff = diffTot/procData.size();
         }
         if (debug) {
            System.out.println("Processing delay = " + (n-this.lastproc));
            System.out.println("Mean Processing delay = " + diff);
         }
         //this.estimatedDelay.add(n-this.lastproc);
         this.estimatedDelay.add(diff);
         //this.totalDelay += (n-this.lastproc);
         this.totalDelay += diff*ptot;
         
         int cease = checkForCeasedArrivals(n);
         if (debug) {
            System.out.println("ceased[" + n + "]=" + cease);
         }
         this.totalCeasedAttempts += cease;
         
         n++;
      }
      System.out.println("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
      System.out.println("Total arrivals = " + this.totalArrivals);
      System.out.println("Total served = " + this.totalServed);
      System.out.println("Total retransmissions = " + this.totalRetransmissions);
      System.out.println("Total server idle count = " + this.serverIdleCount);
      System.out.println("Total remaining power = " + this.totalRemainingPower);
      System.out.println("Total Ceased attempts = " + this.totalCeasedAttempts);
      System.out.println("Total Delay = " + this.totalDelay);
      System.out.println("Mean Delay = " + this.totalDelay/this.totalServed);
      
      for (int i=0; i<6; i++) {
         System.out.print("rt[" + (i+1) + "]=" + this.totalRetSeq[i] + " ");
      }
      System.out.println("\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
   }
   
   public static void main(String[] args) {
      ConvSIP sim = new ConvSIP();
      sim.debug = true;
      sim.init();
      sim.doFluidFlow();
   }

}
